﻿using FTDI.D2xx.WinRT;
using FTDI.D2xx.WinRT.Device;
using FTDI.D2xx.WinRT.Device.EEPROM;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading;
using Windows.System.Threading;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;

namespace TestApp
{
    /// <summary>
    /// An empty page that can be used on its own or navigated to within a Frame.
    /// </summary>
    public sealed partial class MainPage : Page
    {
        private FTManager ftManager;
        private IFTDevice myDevice = null;
        private CancellationTokenSource cancellationTokenSource = new CancellationTokenSource();

        public MainPage()
        {
            this.InitializeComponent();
            String s = FTManager.GetLibraryVersion();
            ftManager = InitializeDriver();
        }

        private FTManager InitializeDriver()
        {
#if CUSTOM_VID_PID
            bool result = FTManager.AddVIDPID(0x1234, 0x4321);
            result = FTManager.RemoveVIDPID(0x0403, 0x6001);
            result = FTManager.RemoveVIDPID(0x0403, 0x6010);
            result = FTManager.RemoveVIDPID(0x0403, 0x6011);
            result = FTManager.RemoveVIDPID(0x0403, 0x6014);
            result = FTManager.RemoveVIDPID(0x0403, 0x6015);
#endif
            return new FTManager();
        }

        public async void StartDevice()
        {
            IList<IFTDeviceInfoNode> deviceList = ftManager.GetDeviceList();

            foreach (IFTDeviceInfoNode deviceInfo in deviceList)
            {
                Debug.WriteLine("Device Type: {0}\r\nSerial Number: {1}\r\nDescription: {2}\r\n\r\n ", 
                    deviceInfo.DeviceType.ToString(), deviceInfo.SerialNumber, deviceInfo.Description);
                if (deviceInfo.DeviceType == DEVICE_TYPE.FT232R)
                {
                    myDevice = ftManager.OpenByDescription(deviceInfo.Description);
                    
                    await myDevice.SetBaudRateAsync(1000000);
                    await myDevice.SetFlowControlAsync(FLOW_CONTROL.RTS_CTS, 0x00, 0x00);
                    await myDevice.SetDataCharacteristicsAsync(WORD_LENGTH.BITS_8, STOP_BITS.BITS_1, PARITY.NONE);
                    await myDevice.SetLatencyTimerAsync(16);

                    var action = ThreadPool.RunAsync(async (source) =>
                    {
                        byte[] dataTx = new byte[10];
                        for (int i = 0; i < dataTx.Length; i++)
                            dataTx[i] = (byte)i;
                        
                        while (!cancellationTokenSource.Token.IsCancellationRequested)
                        {
                            byte[] dataRx = new byte[10];
                            await myDevice.WriteAsync(dataTx, 10);
                            await myDevice.ReadAsync(dataRx, 10);
                        }
                    }, WorkItemPriority.Normal);
                }
            }
        }

        private async void ReadEEPROM()
        {
            if (myDevice != null)
            {
                IFT_EEPROM ee = await myDevice.EepromReadAsync();

                if (myDevice.DeviceInfoNode.DeviceType == DEVICE_TYPE.FT232R)
                {
                    // Cast to the type that corresponds to the device type.
                    FT232R_EEPROM eeData = ee as FT232R_EEPROM;

                    if (eeData == null)
                        return;

                    Debug.WriteLine(@"Manufacturer: {0}\r\nSerial Number: {1}\r\nProduct Description: {2}\r\n\r\n ",
                        eeData.Manufacturer, eeData.SerialNumber, eeData.Product);
                }
            }
        }

        private async void ProgramEEPROM()
        {
            if (myDevice == null)
                return;

            if (myDevice.DeviceInfoNode.DeviceType != DEVICE_TYPE.FT232R)
                return;

            FT232R_EEPROM ee = new FT232R_EEPROM();

            ee.VendorID = 0x0403;
            ee.ProductID = 0x6001;
            ee.LoadVCP = true;
            ee.Manufacturer = "FTDI";
            ee.Product = "FT232R";
            ee.SerialNumber = "FT7654321";
            ee.SerialNumberEnable = true;
            ee.UsbVersion = USB_VERSION.USB_20;
            ee.SelfPowered = false;
            ee.RemoteWakeupEnable = false;
            ee.PullDownEnable = false;
            ee.MaxPower = 500;
            ee.InvertTXD = false;
            ee.InvertRXD = false;
            ee.InvertRTS = false;
            ee.InvertRI = false;
            ee.InvertDTR = false;
            ee.InvertDSR = false;
            ee.InvertDCD = false;
            ee.InvertCTS = false;
            ee.HighIO = false;
            ee.ExternalOscillatorEnable = false;
            ee.CBus4 = FTDI.D2xx.WinRT.Device.EEPROM.FT232R.CBUS_SIGNALS.TXDEN;
            ee.CBus3 = FTDI.D2xx.WinRT.Device.EEPROM.FT232R.CBUS_SIGNALS.TXDEN;
            ee.CBus2 = FTDI.D2xx.WinRT.Device.EEPROM.FT232R.CBUS_SIGNALS.TXDEN;
            ee.CBus1 = FTDI.D2xx.WinRT.Device.EEPROM.FT232R.CBUS_SIGNALS.TXDEN;
            ee.CBus0 = FTDI.D2xx.WinRT.Device.EEPROM.FT232R.CBUS_SIGNALS.TXDEN;

            await myDevice.EepromProgramAsync(ee);
        }

        private void btnListDevices_Click(object sender, RoutedEventArgs e)
        {
            StartDevice();
        }

        private void btnReadEEPROM_Click(object sender, RoutedEventArgs e)
        {
            ReadEEPROM();
        }

        private void btnProgramEEPROM_Click(object sender, RoutedEventArgs e)
        {
            ProgramEEPROM();
        }
    }
}
